program TRIGPOLYNOMIAL;
{--------------------------------------------------------------------}
{  Alg5'5.pas   Pascal program for implementing Algorithm 5.5        }
{                                                                    }
{  NUMERICAL METHODS: Pascal Programs, (c) John H. Mathews 1995      }
{  To accompany the text:                                            }
{  NUMERICAL METHODS for Math., Science & Engineering, 2nd Ed, 1992  }
{  Prentice Hall, Englewood Cliffs, New Jersey, 07632, U.S.A.        }
{  Prentice Hall, Inc.; USA, Canada, Mexico ISBN 0-13-624990-6       }
{  Prentice Hall, International Editions:   ISBN 0-13-625047-5       }
{  This free software is compliments of the author.                  }
{  E-mail address:       in%"mathews@fullerton.edu"                  }
{                                                                    }
{  Algorithm 5.5 (Trigonometric Polynomials).                        }
{  Section   5.4, Fourier Series and Trig. Polynomials, Page 311     }
{--------------------------------------------------------------------}

  uses
    crt;

  const
    MaxN = 361;
    MaxM = 181;
    GNmax = 361;
    Pi = 3.14159265358979323846;
    FunMax = 7;

  type
    Cvector = array[0..MaxM] of real;
    Dvector = array[0..MaxN] of real;
    RVECTOR = array[0..GNmax] of real;
    LETTERS = string[200];
    STATUS = (Computing, Done, Working);

  var
    DNpts, GNpts, Ftype, FunType, Inum, Itype, M, M0, N, Sub: integer;
    T, Xmax, Xmin, Ymax, Ymin: real;
    A0, B0, L, Rnum: real;
    A, B, C: Cvector;
    X, Y: Dvector;
    Xg, Yg: RVECTOR;
    Ans: CHAR;
    Mess: LETTERS;
    Stat, State: STATUS;

  function FX (X: real): real;
  begin
    case FunType of
      1: 
        begin
          if X < -Pi then
            FX := X + 2 * Pi;
          if (-Pi <= X) and (X < 0) then
            FX := 0;
          if (0 <= X) and (X < Pi) then
            FX := X;
          if Pi <= X then
            FX := 0;
        end;
      2: 
        begin
          if X < -Pi then
            FX := 1;
          if (-Pi <= X) and (X < 0) then
            FX := -1;
          if (0 <= X) and (X < Pi) then
            FX := 1;
          if Pi <= X then
            FX := -1;
        end;
      3: 
        begin
          if X < -Pi then
            FX := -3 * Pi / 2 - X;
          if (-Pi <= X) and (X < 0) then
            FX := Pi / 2 + X;
          if (0 <= X) and (X < Pi) then
            FX := Pi / 2 - X;
          if Pi <= X then
            FX := -3 * Pi / 2 + X;
        end;
      4: 
        begin
          if X < -Pi then
            FX := -1;
          if (-Pi <= X) and (X < -Pi / 2) then
            FX := -1;
          if (-Pi / 2 <= X) and (X < Pi / 2) then
            FX := 1;
          if (Pi / 2 <= X) and (X < Pi) then
            FX := -1;
          if Pi <= X then
            FX := -1;
        end;
      5: 
        begin
          if X < -Pi then
            FX := Pi + X;
          if (-Pi <= X) and (X < -Pi / 2) then
            FX := -Pi - X;
          if (-Pi / 2 <= X) and (X < Pi / 2) then
            FX := X;
          if (Pi / 2 <= X) and (X < Pi) then
            FX := Pi - X;
          if Pi <= X then
            FX := -Pi + X;
        end;
      6: 
        begin
          if X < -Pi then
            FX := (X + 2 * Pi) * (X + 2 * Pi) / 4;
          if (-Pi <= X) and (X < Pi) then
            FX := X * X / 4;
          if Pi <= X then
            FX := (X - 2 * Pi) * (X - 2 * Pi) / 4;
        end;
      7: 
        begin
          if X < -Pi then
            FX := X / 2 + Pi;
          if (-Pi <= X) and (X < Pi) then
            FX := X / 2;
          if Pi <= X then
            FX := X / 2 - Pi;
        end;
    end;
  end;

  procedure PRINTFUNCTIONS (FunType: integer);
  begin
    case FunType of
      1: 
        begin
          WRITELN('      /  X + 2*Pi  for          X  < -Pi ');
          WRITELN('     F(X) =   0         for -Pi  <=  X  <  0 ');
          WRITELN('           |  X         for   0  <=  X  <  Pi ');
          WRITELN('           \  0         for  Pi  <=  X ');
        end;
      2: 
        begin
          WRITELN('      /  1  for          X  <= -Pi ');
          WRITELN('     F(X) =  -1  for -Pi  <=  X  <   0 ');
          WRITELN('           |  1  for  0   <=  X  <   Pi ');
          WRITELN('           \ -1  for  Pi  <=  X ');
        end;
      3: 
        begin
          WRITELN('      /  -3*Pi/2 - X  for          X  < -Pi ');
          WRITELN('     F(X) =   Pi/2 + X     for -Pi  <=  X  <  0 ');
          WRITELN('           |  Pi/2 - X     for   0  <=  X  <  Pi ');
          WRITELN('           \  -3Pi/2 + X   for  Pi  <=  X ');
        end;
      4: 
        begin
          WRITELN('      / -1  for             X  < -Pi ');
          WRITELN('           | -1  for    -Pi  <=  X  < -Pi/2 ');
          WRITELN('     F(X) =   1  for  -Pi/2  <=  X  <  Pi/2 ');
          WRITELN('           | -1  for   Pi/2  <=  X  <  Pi ');
          WRITELN('           \ -1  for     Pi  <=  X ');
        end;
      5: 
        begin
          WRITELN('      /  Pi + X  for             X  < -Pi ');
          WRITELN('           | -Pi - X  for    -Pi  <=  X  < -Pi/2 ');
          WRITELN('     F(X) =   X       for  -Pi/2  <=  X  <  Pi/2 ');
          WRITELN('           |  Pi - X  for   Pi/2  <=  X  <  Pi ');
          WRITELN('           \ -Pi + X  for     Pi  <=  X ');
        end;
      6: 
        begin
          WRITELN('      /  (X+2Pi)^2/4  for             X  < -Pi ');
          WRITELN('     F(X) =   X*X/4        for    -Pi  <=  X  <  Pi ');
          WRITELN('           \  (X-2Pi)^2/4  for     Pi  <=  X ');
        end;
      7: 
        begin
          WRITELN('      /  X/2 + Pi  for          X  < -Pi ');
          WRITELN('     F(X) =   X/2       for -Pi  <=  X  <  Pi ');
          WRITELN('           \  X/2 - Pi  for  Pi  <=  X ');
        end;
    end;
  end;

  function F (X: real): real;
    var
      XL, XR: real;
  begin
    XL := X - 0.0000001;
    XR := X + 0.0000001;
    F := (FX(XL) + FX(XR)) / 2;
  end;

  function TP (var A, B: Cvector; M, M0: integer; L, X: real): real;
    var
      J: integer;
      P, Z: real;
  begin
    P := A[0] / 2;
    for J := 1 to M do
      begin
        Z := J * Pi * X / L;
        P := P + A[J] * COS(Z) + B[J] * SIN(Z);
      end;
    if M < M0 then
      begin
        Z := M0 * Pi * X / L;
        P := P + A[M0] * COS(Z) / 2;
      end;
    TP := P;
  end;

  procedure COEFFICIENTS (X, Y: Dvector; var A, B: Cvector; M, M0, N: integer; L: real);
    var
      J, K: integer;
      T: real;
  begin
    for K := 0 to M0 do
      begin
        A[K] := 0;
        B[K] := 0;
      end;
    for K := 1 to N do
      begin
        A[0] := A[0] + Y[K];
        for J := 1 to M0 do
          begin
            T := J * Pi * X[K] / L;
            A[J] := A[J] + Y[K] * COS(T);
            B[J] := B[J] + Y[K] * SIN(T);
          end;
      end;
    for J := 0 to M0 do
      begin
        A[J] := 2 * A[J] / N;
        B[J] := 2 * B[J] / N;
      end;
  end;

  procedure PRINTPOLY (M, M0: integer);
    var
      K, U, V: integer;
  begin
    if (Itype = 1) or (Itype = 2) then
      begin
        case M of
          0: 
            begin
              WRITELN;
              WRITELN('P(x) = a /2');
              WRITELN('        0');
            end;
          1: 
            begin
              WRITELN;
              WRITELN('P(x) = a /2 + a cos(x) + b sin(x)');
              WRITELN('        0      1          1');
            end;
          2: 
            begin
              WRITELN;
              WRITELN('P(x) = a /2 + a cos( x ) + b sin( x ) + a cos( 2x ) + b sin( 2x )');
              WRITELN('        0      1            1            2             2');
            end;
          else
            begin
              WRITELN;
              WRITELN('P(x) = a /2 + a cos( x ) + b sin( x ) +...+ a cos(', M : 2, 'x ) + b sin(', M : 2, 'x )');
              WRITELN('        0      1            1               ', M : 2, '            ', M : 2);
            end;
        end;
        if M < M0 then
          begin
            WRITELN;
            WRITELN;
            WRITELN('     + a  /2 cos(', M0 : 2, 'x )');
            WRITELN('       ', M0 : 2);
          end;
      end;
    if Itype > 2 then
      begin
        case M of
          0: 
            begin
              WRITELN;
              WRITELN('P(x) = a /2');
              WRITELN('        0');
            end;
          1: 
            begin
              WRITELN('                    Pi x          Pi x');
              WRITELN('P(x) = a /2 + a cos(----) + b sin(----)');
              WRITELN('        0      1      L      1      L');
            end;
          2: 
            begin
              WRITELN('                    Pi x          Pi x          2Pi x          2Pi x');
              WRITELN('P(x) = a /2 + a cos(----) + b sin(----) + a cos(-----) + b sin(-----)');
              WRITELN('        0      1      L      1      L      2      L       2      L');
            end;
          else
            begin
              WRITELN('                    Pi x          Pi x              ', M : 2, 'Pi x          ', M : 2, 'Pi x');
              WRITELN('P(x) = a /2 + a cos(----) + b sin(----) +...+ a cos(------) + b sin(------)');
              WRITELN('        0      1      L      1      L         ', M : 2, '      L       ', M : 2, '      L');
            end;
        end;
        if M < M0 then
          begin
            WRITELN;
            WRITELN('                 ', M0 : 2, 'Pi x ');
            WRITELN('     + a  /2 cos(------)');
            WRITELN('       ', M0 : 2, '           L');
          end;
      end;
  end;

  procedure INPUTFUN (var FunType: integer);
    var
      K: integer;
  begin
    CLRSCR;
    FunType := 0;
    while FunType = 0 do
      begin
        CLRSCR;
        for K := 1 to 4 do
          begin
            WRITE('<', K : 1, '>  ');
            PRINTFUNCTIONS(K);
            WRITELN;
          end;
        WRITELN;
        WRITE('     SELECT your function < 1 - ', FunMax : 2, ' >  ');
        FunType := 0;
        READLN(FunType);
        if FunType < 1 then
          FunType := 0;
        if FunType > 4 then
          FunType := 5;
        if FunType = 5 then
          begin
            CLRSCR;
            for K := 5 to 7 do
              begin
                WRITE('<', K : 1, '>  ');
                PRINTFUNCTIONS(K);
                WRITELN;
              end;
            WRITELN;
            WRITE('     SELECT your function < 1 - ', FunMax : 2, ' > ');
            FunType := 0;
            READLN(FunType);
            if FunType < 5 then
              FunType := 0;
            if FunType > 7 then
              FunType := 0;
          end;
      end;
  end;

  procedure GETDATAPOINTS (A0, B0: real; var X, Y: Dvector; var M, M0, N: integer; var L: real; var Itype: integer);
    var
      I, K, MAX: integer;
      H: real;
  begin
    CLRSCR;
    WRITELN;
    WRITELN('        Now you must choose how many abscissas x  , x  ,..., x  are used,');
    WRITELN('                                                0    1        N');
    WRITELN;
    WRITELN('        and the degree of the trigonometric polynomial.');
    WRITELN;
    WRITELN;
    WRITELN('        There are N `periodic` data points.');
    WRITELN;
    WRITELN;
    Mess := '           ENTER  N = ';
    N := 1;
    WRITE(Mess);
    READLN(N);
    if N < 1 then
      N := 1;
    WRITELN;
    WRITELN('        The trigonometric polynomial has degree M.');
    WRITELN;
    WRITELN;
    Mess := '                                         ENTER  M = ';
    M := 1;
    WRITE(Mess);
    READLN(M);
    M0 := M;
    WRITELN;
    MAX := TRUNC((N - 1) / 2);  {TRUNC(INT((N - 1) / 2));}
    if M > MAX then
      begin
        M := MAX;
      end;
    if ODD(N) then  {FRAC(N / 2) <> 0}
      M0 := M
    else if (N < (1 + 2 * M0)) then
      M0 := M + 1
    else
      M0 := M;
    H := (B0 - A0) / N;
    for K := 0 to N - 1 do
      X[K] := A0 + K * H;
    X[N] := B0;
    CLRSCR;
    if M = M0 then
      WRITELN('     The trigonometric polynomial of degree ', M : 2, ' is constructed.');
    if M < M0 then
      WRITELN('     The trigonometric polynomial of degree ', M0 : 2, ' is constructed.');
    WRITELN;
    PRINTPOLY(M, M0);
    WRITELN;
    WRITELN('     Over the interval [ ', A0 : 15 : 7, ' , ', B0 : 15 : 7, '  ]');
    WRITELN;
    case Itype of
      1, 2: 
        WRITELN('     of width 2Pi.');
      3: 
        WRITELN('     of width Pi.');
      4, 5, 6: 
        WRITELN('     where L =', L : 15 : 7);
    end;
    WRITELN;
    WRITELN('     Do you want to enter data points or use  y  = f(x ) ?');
    WRITELN('                                               k      k');
    WRITELN;
    WRITELN;
    WRITELN('     <1>  Enter data points.');
    WRITELN;
    WRITELN('     <2>  Use   y  = f(x ).');
    WRITELN('                 k      k');
    WRITELN;
    Mess := '     Select  < 1 or 2 >  ';
    Ftype := 1;
    WRITE(Mess);
    if M = M0 then
      READLN(Ftype);
    if M < M0 then
      READLN(Ftype);
    if (Ftype = 2) then
      begin
        INPUTFUN(Funtype);
        for K := 0 to N - 1 do
          Y[K] := F(X[K]);
        Y[N] := Y[0];
      end
    else
      begin
        CLRSCR;
        WRITELN;
        case N of
          1: 
            begin
              WRITELN('Enter the ordinate  y . ');
              WRITELN('                     0  ');
            end;
          2: 
            begin
              WRITELN('Enter the ordinates  y  and y .');
              WRITELN('                      0      1 ');
            end;
          else
            begin
              WRITELN('Enter the ', N : 2, ' ordinates  y , y ,...,y  . ');
              WRITELN('                         0   1      ', N - 1 : 0);
            end;
        end;
        WRITELN;
        for K := 0 to N - 1 do
          begin
            WRITELN;
            WRITELN('         x  = ', X[K] : 15 : 7);
            WRITE('          ', K : 0);
            Mess := '         y';
            Y[K] := 0;
            WRITELN;
            WRITE(Mess, K : 1, ' = ');
            READLN(Y[K]);
            WRITELN;
          end;
        Y[N] := Y[0];
      end;
    CLRSCR;
    WRITELN('     Let me think for a while.');
    WRITELN;
    WRITELN('It takes a considerable amount of computing effort to find:');
    WRITELN;
    PRINTPOLY(M, M0);
    if Ftype = 2 then
      begin
        WRITELN;
        WRITELN('For the function:');
        WRITELN;
        WRITE('     ');
        PRINTFUNCTIONS(Funtype);
      end;
  end;

  procedure PRINTCOEFF (A, B: Cvector; M, M0, Itype: integer);
    var
      K: integer;
  begin
    WRITELN;
    WRITE('     A(', 0 : 1, ') = ', A[0] : 15 : 7);
    if Itype > 2 then
      WRITELN('           L = ', L : 15 : 7)
    else
      WRITELN;
    for K := 1 to M do
      begin
        WRITE('     A(', K : 1, ') = ', A[K] : 15 : 7);
        WRITELN('        B(', K : 1, ') = ', B[K] : 15 : 7);
      end;
    if M < M0 then
      WRITELN('     A(', M0 : 1, ') = ', A[M0] : 15 : 7);
  end;

  procedure RESULTS (A, B: Cvector; X, Y: Dvector; M, M0, N: integer; L: real; Itype: integer);
    var
      K: integer;
      Err, XK, PK, YK: real;
  begin
    CLRSCR;
    if Ftype = 2 then
      begin
        WRITELN('The function used was:');
        WRITELN;
        WRITE('     ');
        PRINTFUNCTIONS(Funtype);
      end;
    WRITELN;
    PRINTPOLY(M, M0);
    WRITELN;
    case Itype of
      1: 
        WRITELN('Over the interval [-Pi,Pi]');
      2: 
        WRITELN('Over the interval [0,2Pi]');
      3: 
        WRITELN('Over the interval [0,Pi]');
      4, 5, 6: 
        WRITELN('Over the interval [', A0 : 15 : 7, '  ,', B0 : 15 : 7, '  ].');
    end;
    PRINTCOEFF(A, B, M, M0, Itype);
    WRITELN;
  end;

  procedure EVALUATE (A, B: Cvector; X, Y: Dvector; M, M0, N: integer; L: real; Itype: integer);
    var
      K: integer;
      Err, XK, PK, YK: real;
  begin
    CLRSCR;
    WRITELN;
    WRITELN('     k        x              y              P(x )          Error');
    WRITELN('               k              k                k');
    WRITELN;
    for K := 0 to N do
      begin
        XK := X[K];
        YK := Y[K];
        PK := TP(A, B, M, M0, L, XK);
        Err := YK - PK;
        WRITELN(K : 6, XK : 15 : 7, YK : 15 : 7, PK : 15 : 7, Err : 15 : 7);
      end;
  end;

  procedure MESSAGE;
  begin
    CLRSCR;
    WRITELN('                       TRIGONOMETRIC POLYNOMIALS');
    WRITELN;
    WRITELN;
    WRITELN('     The trigonometric polynomial of degree  M  is constructed:');
    WRITELN;
    WRITELN;
    WRITELN('P(x) = a /2 + a cos(q x) + b sin(q x) +...+ a cos(M q x) + b sin(M q x).');
    WRITELN('        0      1            1                M              M');
    WRITELN;
    WRITELN('It will be the `least squares fit` for the N data points {(x ,y )}.');
    WRITELN('                                                            k  k');
    WRITELN;
    WRITELN('The abscissas are equally spaced over the interval [A,B]:');
    WRITELN;
    WRITELN;
    WRITELN('    x  = a + k(B-A)/N   for  k = 0,1,...,N');
    WRITELN('     k');
    WRITELN;
    WRITELN('The ordinates are periodic:');
    WRITELN;
    WRITELN;
    WRITELN('    y  = y   i.e. they are the same at the left and right endpoints.');
    WRITELN('     N    0');
    WRITELN;
    WRITE('                       Press the <ENTER> key. ');
    READLN(Ans);
    WRITELN;
  end;

  procedure GETINTERVAL (var A0, B0, L: real; var Itype: integer);
    var
      I: integer;
  begin
    CLRSCR;
    WRITELN;
    WRITELN('     You have a choice of the type of interval.');
    WRITELN;
    WRITELN;
    WRITELN('     <1>  [-Pi,Pi]');
    WRITELN;
    WRITELN;
    WRITELN('     <2>  [0,2Pi]');
    WRITELN;
    WRITELN;
    WRITELN('     <3>  [0,Pi]');
    WRITELN;
    WRITELN;
    WRITELN('     <4>  [-L,L]  where you choose L');
    WRITELN;
    WRITELN;
    WRITELN('     <5>  [0,L]   where you choose L');
    WRITELN;
    WRITELN;
    WRITELN('     <6>  [A,B]   where you choose A and B');
    WRITELN;
    WRITELN;
    Mess := '     Select the interval type  < 1 - 6 >  ';
    Itype := 1;
    WRITE(Mess);
    READLN(Itype);
    if Itype < 1 then
      Itype := 1;
    if Itype > 6 then
      Itype := 6;
    CLRSCR;
    for I := 1 to 8 do
      WRITELN;
    case Itype of
      1: 
        begin
          A0 := -Pi;
          B0 := Pi;
          L := Pi;
        end;
      2: 
        begin
          A0 := 0;
          B0 := 2 * Pi;
          L := Pi;
        end;
      3: 
        begin
          A0 := 0;
          B0 := Pi;
          L := Pi / 2;
        end;
      4: 
        begin
          CLRSCR;
          WRITELN;
          WRITELN('          You chose to use an interval');
          WRITELN;
          WRITELN('          of  the  form   [-L,L].');
          WRITELN;
          WRITELN;
          Mess := '                       ENTER  L = ';
          L := Pi;
          WRITE(Mess);
          READLN(L);
          A0 := -L;
          B0 := L;
        end;
      5: 
        begin
          CLRSCR;
          WRITELN;
          WRITELN('          You chose to use an interval');
          WRITELN;
          WRITELN('          of  the  form   [0,L].');
          WRITELN;
          WRITELN;
          Mess := '                      ENTER  L = ';
          L := Pi;
          WRITE(Mess);
          READLN(L);
          A0 := 0;
          B0 := L;
          L := L / 2;
        end;
      6: 
        begin
          CLRSCR;
          WRITELN;
          WRITELN('          You chose to use an interval');
          WRITELN;
          WRITELN('              of   the   form    [A,B].');
          WRITELN;
          WRITELN;
          Mess := '          ENTER the left  endpoint  A = ';
          A0 := 0;
          WRITE(Mess);
          READLN(A0);
          WRITELN;
          Mess := '          ENTER the right endpoint  B = ';
          B0 := 2 * Pi;
          WRITE(Mess);
          READLN(B0);
          L := (B0 - A0) / 2;
        end;
    end;
  end;

begin                                            {Begin Main Program}
  MESSAGE;
  Stat := Working;
  while (Stat = Working) do
    begin
      GETINTERVAL(A0, B0, L, Itype);
      State := Computing;
      while (State = Computing) do
        begin
          GETDATAPOINTS(A0, B0, X, Y, M, M0, N, L, Itype);
          COEFFICIENTS(X, Y, A, B, M, M0, N, L);
          RESULTS(A, B, X, Y, M, M0, N, L, Itype);
          WRITELN;
          WRITE('Do you want to see  a table of values ?  <Y/N>  ');
          READLN(Ans);
          WRITELN;
          if (Ans = 'y') or (Ans = 'Y') then
            EVALUATE(A, B, X, Y, M, M0, N, L, Itype);
          WRITELN;
          WRITE('Want  to change  the number of points ?  <Y/N>  ');
          READLN(Ans);
          WRITELN;
          if (Ans <> 'Y') and (Ans <> 'y') then
            State := Done;
          if (Ans = 'Y') or (Ans = 'y') then
            CLRSCR;
        end;
      WRITELN;
      WRITELN;
      WRITE('Want  to  try  a  different  interval ?  <Y/N>  ');
      READLN(Ans);
      WRITELN;
      if (Ans <> 'Y') and (Ans <> 'y') then
        Stat := Done
    end;
end.                                           {End Main Program}

